/***************************************************************************
 *
 * Copyright (c) 2013,2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <fstream>
#include <iostream>
#include <string>

#include "Log.h"
#include "ButtonSubdivision.h"
#include "CalibrationHelpers.h"
#include "CalibrationTypes.h"
#include "InputDeviceConfiguration.h"
#include "Subdivision.h"
#include "SliderSubdivision.h"
#include "TouchAreaSubdivision.h"

using namespace std;
using namespace LayerManagerCalibration;

static bool compareSubdivisions(const Subdivision* first, const Subdivision* second)
{
    if ((first->getUncalibrated() == true) && (second->getUncalibrated() == false))
    {
        return true;
    }
    else if ((first->getUncalibrated() == false) && (second->getUncalibrated() == true))
    {
        return false;
    }
    else
    {
        return (first->getId() < second->getId());
    }
}

InputDeviceConfiguration::~InputDeviceConfiguration()
{
    list<string*>::iterator iter;

    for (iter = m_DeviceNames.begin();
         iter != m_DeviceNames.end();
         iter++)
    {
        delete *iter;
    }
}

bool InputDeviceConfiguration::parseFile(const string& fileName)
{
    ifstream stream(fileName.c_str());
    string line;
    Subdivision *currentSubdivision = NULL;
    bool inSubdivisions = false;

    if (!stream.good())
    {
        LOG_WARNING("InputDeviceConfiguration",
                    "Device configuration file, name= " << fileName
                    << " not found or not readable");
        return false;
    }

    while (getline(stream, line))
    {
        size_t delimiter = line.find_first_of(':');

        if (delimiter == string::npos)
        {
            LOG_WARNING("InputDeviceConfiguration",
                        "Line found without colon, line=" << line);
            return false;
        }

        string key = line.substr(0, delimiter);
        string value = line.substr(delimiter + 1);

        key = trim(key);
        value = trim(value);

        // Debugging info to be removed
        //cout << "\"" << key << "\" -> \"" << value << "\"" << endl;

        if (key.compare("device name") == 0)
        {
            string* deviceName = new string(value);
            this->m_DeviceNames.push_back(deviceName);
            m_bDeviceNameParsed = true;
        }
        else if (key.compare("calibration file") == 0)
        {
            if (!m_calibration.parseFile(value,true))
            {
                LOG_WARNING("InputDeviceConfiguration",
                            "Failed to parse calibration file, "
                            "name=" << value);
            }
            m_calibrationFileName = value;
        }
        else if (key.compare("subdivisions") == 0)
        {
            inSubdivisions = true;
        }
        else if (key.find_first_of('-') == 0) // list entry (key starts with a '-')
        {
            if (!inSubdivisions)
            {
                LOG_WARNING("InputDeviceConfiguration",
                            "List entry found when not in 'subdivisions'");
                return false;
            }

            if (key.compare("- type") != 0)
            {
                LOG_WARNING("InputDeviceConfiguration",
                            "Subdivision does not start with 'type'");
                return false;
            }

            if (value.compare("button") == 0)
            {
                currentSubdivision = new ButtonSubdivision();
            }
            else if (value.compare("slider") == 0)
            {
                currentSubdivision = new SliderSubdivision();
            }
            else if (value.compare("touch") == 0)
            {
                currentSubdivision = new TouchAreaSubdivision();
            }
            else
            {
                LOG_WARNING("InputDeviceConfiguration",
                            "Found invalid subdivision type, value=" << value);
                return false;
            }

            if (!currentSubdivision->parseStream(stream))
            {
                LOG_WARNING("InputDeviceConfiguration",
                            "Failed to parse subdivision");
                return false;
            }

            if (currentSubdivision->validate())
            {
                m_Subdivisions.push_back(currentSubdivision);
            }
            else
            {
                LOG_WARNING("InputDeviceConfiguration", "Invalid subdivision");
                return false;
            }
        }
    }

    if (!m_bDeviceNameParsed)
    {
        LOG_WARNING("InputDeviceConfiguration", "No device name defined");
        return false;
    }

    m_Subdivisions.sort(compareSubdivisions);

    return true;
}

bool InputDeviceConfiguration::getSubdivisionForCoordinate(const coordinate& raw,
                                                           Subdivision** subdivision,
                                                           coordinate& resultingCoordinate)
{
    list<Subdivision*>::iterator iter;
    coordinate logical;
    this->getCalibration().transformCoordinate(raw, logical);

    if (subdivision != NULL)
        *subdivision = NULL;


    bool foundSubdivision = false;
    for (iter = m_Subdivisions.begin();
         iter != m_Subdivisions.end() && !foundSubdivision;
         iter++)
    {
        Subdivision* currentSubdivision = *iter;
        const coordinate* coord;
        if (currentSubdivision->getUncalibrated() == true)
        {
            coord = &raw;
        }
        else
        {
            coord = &logical;
        }
        if (( coord->x >= currentSubdivision->getTopLeftCoordinate().x ) &&
            ( coord->y >= currentSubdivision->getTopLeftCoordinate().y ) &&
            ( coord->x < currentSubdivision->getBottomRightCoordinate().x ) &&
            ( coord->y < currentSubdivision->getBottomRightCoordinate().y ))
        {
            foundSubdivision = true;
            resultingCoordinate.x = coord->x;
            resultingCoordinate.y = coord->y;

            if (subdivision != NULL)
                *subdivision = currentSubdivision;
        }
    }

    return foundSubdivision;
}

bool InputDeviceConfiguration::getRawCoordinate(const string& subdivisionName,
                                                const coordinate& logical,
                                                coordinate& raw,
                                                uint dispWidth,
                                                uint dispHeight,
                                                bool useCalibration)
{
    bool success = false;

    // get subdivision by name
    Subdivision* subdivision = NULL;
    list<Subdivision*>::const_iterator iter;
    for (iter = getSubdivisions().begin();
         iter != getSubdivisions().end() && subdivision == NULL;
         iter++)
    {
        if (subdivisionName.compare((*iter)->getName()) == 0)
            subdivision = *iter;
    }

    // subdivisionName must be one that exists
    if (subdivision == NULL)
    {
        LOG_WARNING("InputDeviceConfiguration",
                    "No subdivision found matching name="
                    << subdivisionName);
    }
    // Subdivision must not be uncalibrated
    else if (subdivision->getUncalibrated() == true)
    {
        LOG_WARNING("InputDeviceConfiguration",
                    "Subdivision, name="
                    << subdivisionName
                    << " is uncalibrated");
    }
    else
    {
        coordinate coord = {
            logical.x,
            logical.y
        };

        // Touch subdivisions must have their coordinates transformed from
        // display coordinates (pixels) to touch coordinates.
        // We reverse the display offset, add half a pixel (to receive the
        // touch point in the centre of that pixel) then scale according to
        // the display and touch dimensions.
        if (subdivision->getType() == Subdivision::TOUCH)
        {
            TouchAreaSubdivision* touchSubdivision =
                static_cast<TouchAreaSubdivision*>(subdivision);
            coord.x = (coord.x - touchSubdivision->getDisplayOffset().x + 0.5)
                    * subdivision->getWidth() / (float) dispWidth;
            coord.y = (coord.y - touchSubdivision->getDisplayOffset().y + 0.5)
                    * subdivision->getHeight() / (float) dispHeight;
        }

        if (useCalibration)
        {
            success = getCalibration().inverseTransformCoordinate(coord, raw);

            //raw.x = raw.x + subdivision->getTopLeftCoordinate().x;
            //raw.y = raw.y + subdivision->getTopLeftCoordinate().y;
        }
        else
        {
            //raw.x = coord.x + subdivision->getTopLeftCoordinate().x;
            //raw.y = coord.y + subdivision->getTopLeftCoordinate().y;
            success = true;
        }
    }
    return success;
}
